<?php

namespace App\Http\Controllers;

use App\Models\Asset;
use App\Models\PackingList;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\View\View;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class PackingListController extends Controller
{
    /**
     * Menampilkan halaman utama Barang Keluar & Packing List.
     */
    public function viewPage(): View
    {
        $data = [];
        return view('backend.packing_lists.index', compact('data'));
    }

    /**
     * Mengekspor packing list ke format PDF.
     */
    public function exportPdf(string $id): Response
    {
        // PERBAIKAN: Gunakan relasi 'creator' BUKAN 'user' jika relasinya creator()
        // Sesuaikan 'creator' jika nama relasi di Model PackingList berbeda
        $packingList = PackingList::with('creator', 'assets')->findOrFail($id);
        $pdf = Pdf::loadView('backend.packing_lists.pdf', ['packingList' => $packingList]);
        return $pdf->stream($packingList->document_number . '.pdf');
    }

    // ===================================================================
    // API METHODS
    // ===================================================================

    /**
     * API: Mengambil riwayat packing list dengan paginasi dan filter.
     */
    public function index(Request $request): JsonResponse
    {
        // ... (Method index tidak berubah) ...
        $query = PackingList::with('creator:id,name')->withCount('assets');

        $query->when($request->input('search'), function ($q, $search) {
            $q->where('document_number', 'like', "%{$search}%")
                ->orWhere('recipient_name', 'like', "%{$search}%");
        });

        $packingLists = $query->latest()->paginate($request->input('perPage', 10));
        return response()->json($packingLists);
    }

    /**
     * API: Menyimpan data packing list baru.
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'recipient_name' => 'required|string|max:100',
            'notes' => 'nullable|string',
            'asset_ids' => 'required|array|min:1',
            'asset_ids.*' => 'exists:assets,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        try {
            DB::transaction(function () use ($request) {
                // Logika pembuatan nomor dokumen unik
                $prefix = 'PL-' . date('Ymd') . '-';
                $lastEntry = PackingList::where('document_number', 'like', "{$prefix}%")->orderBy('document_number', 'desc')->first();
                $nextNumber = $lastEntry ? ((int) substr($lastEntry->document_number, -4)) + 1 : 1;
                $documentNumber = $prefix . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);

                // --- >>> PERUBAHAN DI SINI <<< ---
                $user = Auth::user(); // Ambil user yang sedang login
                Log::info('User Signature Path: ' . $user->signature_image);

                $packingList = PackingList::create([
                    'document_number' => $documentNumber,
                    'recipient_name' => $request->input('recipient_name'),
                    'notes' => $request->input('notes'),
                    'created_by' => $user->id, // Gunakan ID user yang login
                    // Langsung salin path TTD user ke signature_pad saat create
                    'signature_pad' => $user->signature_image ?? null
                ]);
                // --- >>> AKHIR PERUBAHAN <<< ---

                // Menghubungkan aset dan update status/stok
                $packingList->assets()->attach($request->input('asset_ids'));
                $assets = Asset::find($request->input('asset_ids'));
                foreach ($assets as $asset) {
                    if ($asset->asset_type == 'fixed_asset') {
                        // PERBAIKAN: Ubah status menjadi 'disposed' karena Packing List = Barang Keluar Permanen
                        $asset->update(['status' => 'disposed']);
                    } else {
                        if ($asset->current_stock > 0) {
                            $asset->decrement('current_stock');
                            // Opsional: Cek stok minimum setelah decrement
                            // if ($asset->current_stock <= $asset->minimum_stock) {
                            //     // Kirim notifikasi stok menipis
                            // }
                        }
                    }
                }
            });

            return response()->json(['message' => 'Packing list berhasil dibuat!'], 201);
        } catch (\Exception $e) {
            // Log error untuk debugging
            Log::error('Error creating packing list: ' . $e->getMessage());
            return response()->json(['message' => 'Terjadi kesalahan saat membuat packing list.'], 500);
        }
    }

    /**
     * API: Mengambil daftar aset yang tersedia untuk Select2.
     */
    public function getAvailableAssets(Request $request): JsonResponse
    {
        $search = $request->input('q');

        // DEBUG: Log parameter pencarian
        Log::info('PackingList - Asset Search', [
            'search' => $search,
            'request_all' => $request->all()
        ]);

        $assets = Asset::query()
            ->where('status', 'available')
            // Filter sudah dihapus agar Fixed Asset juga muncul
            ->where('current_stock', '>', 0)   // yang stoknya lebih dari 0
            ->when($search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('name_asset', 'like', "%{$search}%")
                        ->orWhere('serial_number', 'like', "%{$search}%");
                });
            })
            ->orderBy('name_asset')
            ->limit(20)
            ->get(['id', 'name_asset', 'serial_number', 'asset_type', 'current_stock']);

        // DEBUG: Log hasil query
        Log::info('PackingList - Assets Found', [
            'total' => $assets->count(),
            'fixed_assets' => $assets->where('asset_type', 'fixed_asset')->count(),
            'consumables' => $assets->where('asset_type', 'consumable')->count(),
            'assets' => $assets->map(fn($a) => [
                'id' => $a->id,
                'name' => $a->name_asset,
                'type' => $a->asset_type,
                'stock' => $a->current_stock
            ])->toArray()
        ]);

        $formattedAssets = $assets->map(function ($asset) {
            // Format berbeda untuk Fixed Asset dan Consumable
            if ($asset->asset_type === 'fixed_asset') {
                // Fixed Asset: tampilkan dengan Serial Number
                $text = $asset->name_asset . ' [SN: ' . ($asset->serial_number ?? 'N/A') . ']';
            } else {
                // Consumable: tampilkan dengan Stok
                $text = $asset->name_asset . ' (Stok: ' . $asset->current_stock . ')';
            }
            
            return ['id' => $asset->id, 'text' => $text];
        });

        return response()->json(['results' => $formattedAssets]);
    }
}
